//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	IWP 6
// Objective: 	Part 6 of IWP work - calculating children's percentile ranks, estimating IWP and attributing a share to inheritances
//
//==============================================================================

clear all

************************
*** Data import ********
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

drop if hgint == 0 //drop non-responders

merge m:1 xwaveid wave using individual_weight_vars_restricted.dta //individual wealth variables 
drop if _merge == 2
drop _merge

merge m:1 xwaveid wave using IWP_data_5.dta //parent wealth percentiles
drop if _merge == 2
drop _merge

gen flag = 1
merge m:1 flag using IWP_data_6.dta
drop _merge

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

merge m:1 wave using CPI.dta //CPI data
drop if _merge == 2
drop _merge

xtset xwaveid wave

drop if wlra_s == 0 //create balanced panel from waves 1-19, to match IWP 1.do

******************************
*** Create basic variables ***
******************************

gen child_net_worth_HH = (hwnwip - hwnwin)*100 / CPI //child household net worth in real terms

gen year = wave + 2000 //year

gen hgage_squ = hgage^2

gen age_63_73 = 0 //flag for child cohort
replace age_63_73 = 1 if (year - hgage) <= 1954 & (year - hgage) >= 1944

local samples 2pc 10pc 20pc //spread parent wealth percentiles and sample flags across all waves
foreach x of local samples{
	egen m_net_worth_couple_pt_`x' = max(net_worth_couple_ptile_`x'), by(xwaveid)
	egen m_net_worth_couple_exp_pt_`x' = max(net_worth_couple_exp_ptile_`x'), by(xwaveid)
	egen max_sample_`x'_zero = max(sample_`x'_zero), by(xwaveid)
}

egen max_yob_male = max(yob_male), by(xwaveid) //spread year of birth and year of birth squared flags across all waves
egen max_yob_female = max(yob_female), by(xwaveid)
egen max_yob_male_squ = max(yob_male_squ), by(xwaveid)
egen max_yob_female_squ = max(yob_female_squ), by(xwaveid)

*********************************************************
*** Create child HH wealth with inheritances removed  ***
*********************************************************

gen bequest = oifinha //inheritances
replace bequest = 0 if bequest == -1
replace bequest = . if bequest < -1
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due to HH deaths

egen bequest_HH = total(bequest), by(hhrhid wave) //sum of all inheritances received by HH in each year
replace bequest_HH = bequest_HH*100 / CPI //convert to real

codebook bequest_HH

gen child_net_worth_HH_bequest = child_net_worth_HH //central estimate
forvalues x = 0/17{
	replace child_net_worth_HH_bequest = child_net_worth_HH_bequest - bequest_share_remaining`x' * L`x'.bequest_HH
}

gen child_net_worth_HH_bequest_fast = child_net_worth_HH //depletion twice as fast as expected
forvalues x = 0/17{
	replace child_net_worth_HH_bequest_fast = child_net_worth_HH_bequest_fast - bequest_share_remaining_fast`x' * L`x'.bequest_HH
}

gen child_net_worth_HH_bequest_slow = child_net_worth_HH //no depletion
forvalues x = 0/17{
	replace child_net_worth_HH_bequest_slow = child_net_worth_HH_bequest_slow - bequest_share_remaining_slow`x' * L`x'.bequest_HH
}

************************************************
*** Create child HH wealth percentile ranks  ***
************************************************

local samples 2pc 10pc 20pc
foreach x of local samples{
xtile child_pt_`x' = child_net_worth_HH [pweight = wlra_s] if wave == 18 & max_sample_`x'_zero == 1, n(100)
xtile child_pt_bequest_`x' = child_net_worth_HH_bequest [pweight = wlra_s] if wave == 18 & max_sample_`x'_zero == 1, n(100)
xtile child_pt_bequest_fast_`x' = child_net_worth_HH_bequest_fast [pweight = wlra_s] if wave == 18 & max_sample_`x'_zero == 1, n(100)
xtile child_pt_bequest_slow_`x' = child_net_worth_HH_bequest_slow [pweight = wlra_s] if wave == 18 & max_sample_`x'_zero == 1, n(100)
}

drop if wave != 18

*********************************************************
*** Estimate IWP and attribute share to inheritances  ***
*********************************************************
*Note, parent wealth is censored at 0, so regressions are run only on the non-zero observations. This should not cause bias, as the censoring is exogenous w.r.t. child wealth (see http://web.mit.edu/tstoker/www/Rigobon_Stoker_IER_June_07.pdf)

*All results in table B.12

reg child_pt_2pc m_net_worth_couple_pt_2pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_2pc_zero == 1 & m_net_worth_couple_pt_2pc > 2 //.2399531
reg child_pt_bequest_2pc m_net_worth_couple_pt_2pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_2pc_zero == 1 & m_net_worth_couple_pt_2pc > 2 //share attributed to inheritances = 1 - .1571181 / .2399531 = 35%
reg child_pt_bequest_fast_2pc m_net_worth_couple_pt_2pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_2pc_zero == 1 & m_net_worth_couple_pt_2pc > 2 //share attributed to inheritances = 1 - .1914919 / .2399531 = 20%
reg child_pt_bequest_slow_2pc m_net_worth_couple_pt_2pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_2pc_zero == 1 & m_net_worth_couple_pt_2pc > 2 //share attributed to inheritances = 1 - .0992658 / .2399531 = 59%

reg child_pt_10pc m_net_worth_couple_pt_10pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_10pc_zero == 1 & m_net_worth_couple_pt_10pc > 10 //.2618011 
reg child_pt_bequest_10pc m_net_worth_couple_pt_10pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_10pc_zero == 1 & m_net_worth_couple_pt_10pc > 10 //share attributed to inheritances = 1 - .167315 / .2618011 = 36%
reg child_pt_bequest_fast_10pc m_net_worth_couple_pt_10pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_10pc_zero == 1 & m_net_worth_couple_pt_10pc > 10  //share attributed to inheritances = 1 - .2089583 / .2618011 = 20%
reg child_pt_bequest_slow_10pc m_net_worth_couple_pt_10pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_10pc_zero == 1 & m_net_worth_couple_pt_10pc > 10  //share attributed to inheritances = 1 - .108424 / .2618011 = 59%

reg child_pt_20pc m_net_worth_couple_pt_20pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_20pc_zero == 1 & m_net_worth_couple_pt_20pc > 20 //.2974159
reg child_pt_bequest_20pc m_net_worth_couple_pt_20pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_20pc_zero == 1 & m_net_worth_couple_pt_20pc > 20 //share attributed to inheritances = 1 - .1915806 / .2974159 = 36%
reg child_pt_bequest_fast_20pc m_net_worth_couple_pt_20pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_20pc_zero == 1 & m_net_worth_couple_pt_20pc > 20 //share attributed to inheritances = 1 - .2392223 / .2974159 = 20%
reg child_pt_bequest_slow_20pc m_net_worth_couple_pt_20pc max_yob_male max_yob_male_squ max_yob_female max_yob_female_squ hgage hgage_squ  [pweight = wlra_s] if max_sample_20pc_zero == 1 & m_net_worth_couple_pt_20pc > 20 //share attributed to inheritances = 1 - .1255256 / .2974159 = 58%

keep if max_sample_10pc_zero == 1
keep child_pt_10pc m_net_worth_couple_pt_10pc wlra_s
xtile parent_quintile = m_net_worth_couple_pt_10pc [pweight = wlra_s], n(5)

export delimited IWP_box_plot.csv, replace //box plot is created using R














